/*****************************************************************************
 * prodos/file.c
 * File and inode operations for regular files.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 26-May-2001: Created
 *****************************************************************************/

/*= Kernel Includes =========================================================*/


#include <linux/types.h>
#include <linux/locks.h>

/*= ProDOS Includes =========================================================*/

#include "prodos.h"

/*= Forward Declarations ====================================================*/

/* For prodos_address_space_operations. */
int prodos_readpage(struct file *,struct page *);
int prodos_writepage(struct page *);
int prodos_prepare_write(struct file *,struct page *,unsigned int, unsigned int);
int prodos_commit_write(struct file*,struct page *,unsigned int,unsigned int);
int prodos_bmap(struct address_space *,long);

/*= VFS Interface Structures ================================================*/

struct inode_operations prodos_file_inode_operations = {

};

struct file_operations prodos_file_operations = {
	read: generic_file_read,
	write: generic_file_write,
	mmap: generic_file_mmap
};

struct address_space_operations prodos_address_space_operations = {
	readpage: prodos_readpage,
	writepage: prodos_writepage,
	sync_page: block_sync_page,
	prepare_write: prodos_prepare_write,
	commit_write: prodos_commit_write,
	bmap: prodos_bmap
};

/*= Private Structures ======================================================*/

/* Items defining the structure of the first block (header) of a meta file. */
#define PRODOS_META_MAGIC				0x504d5441
#define PRODOS_META_WARNING				"\nWARNING!!! DO NOT EDIT THIS FILE!!!\n\n"
struct prodos_meta_header {
	u32 magic;
	u8 warning[490];
	u8 file_type;
	u16 aux_type;
	u8 access;
	u32 data_eof;
	u32 res_eof;
} __attribute__((packed));

/*= Interface Functions =====================================================*/

/*****************************************************************************
 * prodos_get_block()
 * Translate a block number within an @inode into a logical disk block number.
 *****************************************************************************/
int prodos_get_block(struct inode *inode,long block,struct buffer_head *bh_res,int create) {
	SHORTCUT struct prodos_inode_info * const pi = PRODOS_I(inode);
	SHORTCUT struct super_block * const sb = inode->i_sb;
	SHORTCUT const u16 dfrk = PRODOS_INO_DFORK(inode->i_ino);
	int result = 0;
	u16 blk = pi->i_key;
	u8 stype = pi->i_stype;
	struct buffer_head *bh = NULL;

	/* Verify that the block number is valid. */
	if ((block < 0) || (block >= inode->i_blocks)) {
		PRODOS_ERROR_1(sb,"bad block number: %i",(int)block);
		result = -EIO;
		goto out_cleanup;
	}

	/* Need to dig a little deeper for "extended" files. */
	if (stype == PRODOS_STYPE_EXTENDED) {
		struct prodos_ext_key_block *ext_key = NULL;
		struct prodos_fork_entry *fork_ent = NULL;

		/* Load the extended directory block. */
		bh = prodos_bread(sb,blk);
		if (!bh) {
			result = -EIO;
			goto out_cleanup;
		}
		ext_key = (struct prodos_ext_key_block*)bh->b_data;

		/* Meta files require some special processing. */
		if (dfrk == PRODOS_DFORK_META) {
			/* First block is header. */
			if (!block) {
				struct prodos_meta_header meta_hdr;

				/* Fill the meta file header. */
				memset(&meta_hdr,0,sizeof(meta_hdr));
				meta_hdr.magic = cpu_to_be32(PRODOS_META_MAGIC);
				strcpy(meta_hdr.warning,PRODOS_META_WARNING);
				meta_hdr.file_type = pi->i_filetype;
				meta_hdr.aux_type = cpu_to_le16(pi->i_auxtype);
				meta_hdr.access = pi->i_access;
				meta_hdr.data_eof = cpu_to_le32(le24_to_cpup(&ext_key->data.eof));
				meta_hdr.res_eof = cpu_to_le32(le24_to_cpup(&ext_key->res.eof));

				/* Copy the header data into bh_res and signal that the buffer
				 should be returned unmodified. */
				lock_buffer(bh_res);
				memcpy(bh_res->b_data,&meta_hdr,sizeof(meta_hdr));
				bh_res->b_state |= (1 << BH_Mapped);
				mark_buffer_uptodate(bh_res,1);
				unlock_buffer(bh_res);
				goto out_cleanup;
			}

			/* Get appropriate fork and translate 'block' to a block index
			 within that fork. */
			if (block <= PRODOS_DATA_BLOCKS_REQ(le24_to_cpup(&ext_key->data.eof))) {
				fork_ent = &ext_key->data;
				block--;
			}
			else {
				fork_ent = &ext_key->res;
				block -= (1 + PRODOS_DATA_BLOCKS_REQ(le24_to_cpup(&ext_key->data.eof)));
			}
		}

		/* Regular fork files are simpler. */
		else if (dfrk == PRODOS_DFORK_DATA) fork_ent = &ext_key->data;
		else fork_ent = &ext_key->res;

		/* Copy fork info for later (byte order also handled later.) */
		stype = fork_ent->stype << 4;
		blk = fork_ent->key_block;

		/* Release the extended directory block. */
		prodos_brelse(bh);
		bh = NULL;
	}

	/* Handle meta files for NON-extended files. */
	else if (dfrk == PRODOS_DFORK_META) {
		/* First block is header. */
		if (!block) {
			struct prodos_meta_header meta_hdr;

			/* Fill the meta file header. */
			memset(&meta_hdr,0,sizeof(meta_hdr));
			meta_hdr.magic = cpu_to_be32(PRODOS_META_MAGIC);
			strcpy(meta_hdr.warning,PRODOS_META_WARNING);
			meta_hdr.file_type = pi->i_filetype;
			meta_hdr.aux_type = cpu_to_le16(pi->i_auxtype);
			meta_hdr.access = pi->i_access;
			meta_hdr.data_eof = cpu_to_le32(inode->i_size);
			meta_hdr.res_eof = 0xffffffff;

			/* Copy the header data into bh_res and signal that the buffer
			 should be returned unmodified. */
			lock_buffer(bh_res);
			memcpy(bh_res->b_data,&meta_hdr,sizeof(meta_hdr));
			bh_res->b_state |= (1 << BH_Mapped);
			mark_buffer_uptodate(bh_res,1);
			unlock_buffer(bh_res);
			goto out_cleanup;
		}

		/* Calculate actual block number in the file. */
		block--;
	}

	/* Appropriate action depends upon the file's storage type. */
	switch (stype) {
		case PRODOS_STYPE_TREE:
			/* Load double indirect block. */
			bh = prodos_bread(sb,blk);
			if (!bh) {
				result = -EIO;
				goto out_cleanup;
			}

			/* Get indirect block and release double indirect block. */
			blk = PRODOS_GET_INDIRECT_ENTRY(bh->b_data,block >> 8);
			prodos_brelse(bh);
			bh = NULL;
			/* Fall through. */
		case PRODOS_STYPE_SAPLING:
			/* Load indirect block. */
			bh = prodos_bread(sb,blk);
			if (!bh) {
				result = -EIO;
				goto out_cleanup;
			}

			/* Get data block and release indirect block. */
			blk = PRODOS_GET_INDIRECT_ENTRY(bh->b_data,block & 0xff);
			prodos_brelse(bh);
			bh = NULL;
			/* Fall through. */
		case PRODOS_STYPE_SEEDLING:
			/* Check for a sparse block. */
			if (!blk) {
				/* Mark buffer up-to-date without setting BH_Mapped flag. This
				 will be read as a block full of zeroes. */
				mark_buffer_uptodate(bh_res,1);
				goto out_cleanup;
			}

			/* Set bh_res fields to cause the block to be read from disk. */
			bh_res->b_blocknr = PRODOS_SB(sb)->s_part_start + blk;
			bh_res->b_dev = inode->i_dev;
			bh_res->b_state |= (1 << BH_Mapped);
	}

out_cleanup:
	if (bh) prodos_brelse(bh);
	return result;
}

/*****************************************************************************
 * prodos_get_block_write()
 * get_block_t function for write operations.  This function is ugly and
 * probably temporary.  Ultimately it should be merged with prodos_get_block()
 * and thoroughly cleaned.  What you see here is basically a quick and (very)
 * dirty hack.
 *****************************************************************************/
int prodos_get_block_write(struct inode *inode,long block,struct buffer_head *bh_res,int create) {
	SHORTCUT struct prodos_inode_info * const pi = PRODOS_I(inode);
	int result = 0;

	/* Croak on non-regular files for now. */
	if ((pi->i_stype < PRODOS_STYPE_SEEDLING) || (pi->i_stype > PRODOS_STYPE_TREE)) {
		PRODOS_ERROR_0(inode->i_sb,"extended files not supported for writing");
		result = -EPERM;
		goto out_cleanup;
	}

	/* Can't translate block numbers that are out of range. */
	if (block >= PRODOS_MAX_FILE_SIZE >> PRODOS_BLOCK_SIZE_BITS) {
		result = -EFBIG;
		goto out_cleanup;
	}

	/* Expand the file if necessary. */
	if (block >= inode->i_blocks) {
		u8 new_stype = pi->i_stype;
		u16 new_key = pi->i_key;
		u16 new_bused = inode->i_blocks;

		/* May have to convert a seedling into a sapling. */
		if ((block > 0) && (new_stype == PRODOS_STYPE_SEEDLING)) {
			u16 key_ptr = 0;
			struct buffer_head *bh = NULL;

			/* Allocate the new indirect block. */
			key_ptr = prodos_alloc_block(inode->i_sb);
			if (!key_ptr) {
				result = -ENOSPC;
				goto out_cleanup;
			}

			/* Load and initialize the indirect block. */
			bh = prodos_bread(inode->i_sb,key_ptr);
			if (!bh) {
				result = -EIO;
				goto out_cleanup;
			}
			memset(bh->b_data,0,PRODOS_BLOCK_SIZE);
			PRODOS_SET_INDIRECT_ENTRY(bh->b_data,0,new_key);
			mark_buffer_dirty(bh);
			prodos_brelse(bh);

			new_stype = PRODOS_STYPE_SAPLING;
			new_key = key_ptr;
			new_bused++;
		}

		/* May have to convert a sapling into a tree. */
		if ((block > 256) && (new_stype == PRODOS_STYPE_SAPLING)) {
			u16 key_ptr = 0;
			struct buffer_head *bh = NULL;

			/* Allocate the new double indirect block. */
			key_ptr = prodos_alloc_block(inode->i_sb);
			if (!key_ptr) {
				result = -ENOSPC;
				goto out_cleanup;
			}

			/* Load and initialize the double indirect block. */
			bh = prodos_bread(inode->i_sb,key_ptr);
			if (!bh) {
				result = -EIO;
				goto out_cleanup;
			}
			memset(bh->b_data,0,PRODOS_BLOCK_SIZE);
			PRODOS_SET_INDIRECT_ENTRY(bh->b_data,0,new_key);
			mark_buffer_dirty(bh);
			prodos_brelse(bh);

			new_stype = PRODOS_STYPE_TREE;
			new_key = key_ptr;
			new_bused++;
		}

		/* Update the inode if storage type changed. */
		if (new_stype != pi->i_stype) {
			pi->i_stype = new_stype;
			pi->i_key = new_key;
			inode->i_blocks = new_bused;
			mark_inode_dirty(inode);
		}

		/* May need to expand indexes for a tree file. */
		if (new_stype == PRODOS_STYPE_TREE) {
			u16 ind_existing = ((inode->i_blocks - 1) >> 8) + 1;
			u16 ind_needed = block >> 8;
			u16 ind_ptr = 0;
			struct buffer_head *dind_bh = NULL;
			struct buffer_head *ind_bh = NULL;

			dind_bh = prodos_bread(inode->i_sb,pi->i_key);
			if (!dind_bh) {
				result = -EIO;
				goto out_cleanup;
			}
			for (;ind_existing < ind_needed;ind_existing++) {
				ind_ptr = prodos_alloc_block(inode->i_sb);
				if (!ind_ptr) {
					result = -ENOSPC;
					goto out_cleanup;
				}
				ind_bh = prodos_bread(inode->i_sb,ind_ptr);
				if (!ind_bh) {
					result = -EIO;
					goto out_cleanup;
				}
				memset(ind_bh,0,PRODOS_BLOCK_SIZE);
				PRODOS_SET_INDIRECT_ENTRY(dind_bh->b_data,ind_existing,ind_ptr);
				prodos_brelse(ind_bh);
				inode->i_blocks = ++new_bused;
			}
			prodos_brelse(dind_bh);
		}

	}

	{
		u16 blk = pi->i_key;
		struct buffer_head *bh = NULL;
		/* Appropriate action depends upon the file's storage type. */
		switch (pi->i_stype) {
			case PRODOS_STYPE_TREE:
				/* Load double indirect block. */
				bh = prodos_bread(inode->i_sb,blk);
				if (!bh) {
					result = -EIO;
					goto out_cleanup;
				}

				/* Get indirect block and release double indirect block. */
				blk = PRODOS_GET_INDIRECT_ENTRY(bh->b_data,block >> 8);
				prodos_brelse(bh);
				bh = NULL;
				/* Fall through. */
			case PRODOS_STYPE_SAPLING:
				/* Load indirect block. */
				bh = prodos_bread(inode->i_sb,blk);
				if (!bh) {
					result = -EIO;
					goto out_cleanup;
				}

				/* Get data block and release indirect block. */
				blk = PRODOS_GET_INDIRECT_ENTRY(bh->b_data,block & 0xff);
				if (!blk) {
					bh_res->b_state |= (1UL << BH_New);
					blk = prodos_alloc_block(inode->i_sb);
					if (!blk) {
						result = -ENOSPC;
						goto out_cleanup;
					}
					PRODOS_SET_INDIRECT_ENTRY(bh->b_data,block & 0xff,blk);
					inode->i_blocks++;
				}
				prodos_brelse(bh);
				bh = NULL;

				/* Fall through. */
			case PRODOS_STYPE_SEEDLING:
				/* Set bh_res fields to cause the block to be read from disk. */
				bh_res->b_blocknr = PRODOS_SB(inode->i_sb)->s_part_start + blk;
				bh_res->b_dev = inode->i_dev;
				bh_res->b_state |= (1 << BH_Mapped);
		}
	}

	mark_inode_dirty(inode);

out_cleanup:
	return result;
}

/*****************************************************************************
 * prodos_readpage()
 *****************************************************************************/
int prodos_readpage(struct file *filp,struct page *page) {
	int result = 0;

	/* Let the generic "read page" function do most of the work. */
	result = block_read_full_page(page,prodos_get_block);

	/* Perform CR->LF conversion for simple text files. */
	if (PRODOS_I(filp->f_dentry->d_inode)->i_flags & PRODOS_I_CRCONV) {
		wait_on_page(page);
		if (Page_Uptodate(page)) {
			int i = 0;
			char *pdata = kmap(page);
			for (i = 0;i < PAGE_CACHE_SIZE;i++,pdata++)
				if (*pdata == '\r') *pdata = '\n';
			kunmap(page);
		}
	}

	/* Propagate block_read_full_page() return value. */
	return result;
}

/*****************************************************************************
 * prodos_writepage()
 *****************************************************************************/
int prodos_writepage(struct page *page) {
	/* Generic function does the work with the help of a get_block function. */
	PRODOS_ERROR_0(page->mapping->host->i_sb,"called");
	return block_write_full_page(page,prodos_get_block_write);
}


/*****************************************************************************
 * prodos_prepare_write()
 *****************************************************************************/
int prodos_prepare_write(struct file *file,struct page *page,unsigned int from,unsigned int to) {
	return block_prepare_write(page,from,to,prodos_get_block_write);
}

/*****************************************************************************
 * prodos_commit_write()
 *****************************************************************************/
int prodos_commit_write(struct file *file,struct page *page,unsigned int from,unsigned int to) {
	/* Convert text files to ProDOS format. */
	if (PRODOS_I(page->mapping->host)->i_flags & PRODOS_I_CRCONV) {
		int i = 0;
		char *pdata = kmap(page);
		for (i = 0;i < PAGE_CACHE_SIZE;i++,pdata++)
			if (*pdata == '\n') *pdata = '\r';
		kunmap(page);
	}

	/* Let generic function do the real work. */
	return generic_commit_write(file,page,from,to);
}

/*****************************************************************************
 * prodos_bmap()
 *****************************************************************************/
int prodos_bmap(struct address_space *mapping,long block) {
	/* Generic function does the work with the help of a get_block function. */
	PRODOS_ERROR_0(mapping->host->i_sb,"called");
	return generic_block_bmap(mapping,block,prodos_get_block_write);
}




